/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 * Copyright 2022 The Quilt Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.impl.resource.loader;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.quiltmc.qsl.resource.loader.api.ResourceLoader;
import net.fabricmc.fabric.api.resource.IdentifiableResourceReloadListener;
import net.fabricmc.fabric.api.resource.ResourceManagerHelper;
import net.minecraft.class_2960;
import net.minecraft.class_3264;

public class ResourceManagerHelperImpl implements ResourceManagerHelper {
	private static final Map<class_3264, ResourceManagerHelperImpl> registryMap = new HashMap<>();

	private final class_3264 type;

	private class_2960 lastResourceReloaderId = null;

	private ResourceManagerHelperImpl(class_3264 type) {
		this.type = type;
	}

	public static ResourceManagerHelperImpl get(class_3264 type) {
		return registryMap.computeIfAbsent(type, ResourceManagerHelperImpl::new);
	}

	@Override
	public void registerReloadListener(IdentifiableResourceReloadListener listener) {
		var resourceLoader = ResourceLoader.get(this.type);
		resourceLoader.registerReloader(listener);

		// Inject a synthetic ordering between listeners registered on the same namespace that are registered after each other
		// This matches the existing behavior of fabric-api where listeners are called in registration order, fixing some compatibility issues.
		// We split on namespaces to prevent grouping all fabric based listeners into one long chain and causing potential issues in actual ordering.
		// see i.e https://gitlab.com/cable-mc/cobblemon/-/issues/148 or https://github.com/apace100/calio/issues/3
		if (
				this.lastResourceReloaderId != null
						&& Objects.equals(this.lastResourceReloaderId.method_12836(), listener.getQuiltId().method_12836())
		) {
			resourceLoader.addReloaderOrdering(this.lastResourceReloaderId, listener.getQuiltId());
		}

		// Reimplement getFabricDependencies' functionality with reloader reordering
		for (var dependency : listener.getFabricDependencies()) {
			resourceLoader.addReloaderOrdering(dependency, listener.getQuiltId());
		}

		this.lastResourceReloaderId = listener.getQuiltId();
	}
}
