/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 * Copyright 2023 The Quilt Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.object.builder.v1.block.type;

import net.minecraft.class_2498;
import net.minecraft.class_2960;
import net.minecraft.class_3414;
import net.minecraft.class_3417;
import net.minecraft.class_4719;
import net.minecraft.class_8177;

/**
 * This class allows easy creation of {@link class_4719}s.
 *
 * <p>A {@link class_4719} is used to tell the game what textures signs should use, as well as sounds for both signs and fence gates.
 *
 * <p>Regular sign textures are stored at {@code [namespace]/textures/entity/signs/[path].png}.
 * <br>Hanging sign textures are stored at {@code [namespace]/textures/entity/signs/hanging/[path].png}.
 *
 * @see BlockSetTypeBuilder
 */
public final class WoodTypeBuilder {
	private class_2498 soundGroup = class_2498.field_11547;
	private class_2498 hangingSignSoundGroup = class_2498.field_40313;
	private class_3414 fenceGateCloseSound = class_3417.field_14861;
	private class_3414 fenceGateOpenSound = class_3417.field_14766;

	/**
	 * Sets this wood type's sound group.
	 *
	 * <p>Defaults to {@link class_2498#field_11547}.
	 *
	 * @return this builder for chaining
	 */
	public WoodTypeBuilder soundGroup(class_2498 soundGroup) {
		this.soundGroup = soundGroup;
		return this;
	}

	/**
	 * Sets this wood type's hanging sign sound group.
	 *
	 * <p>Defaults to {@link class_2498#field_40313}.
	 *
	 * @return this builder for chaining
	 */
	public WoodTypeBuilder hangingSignSoundGroup(class_2498 hangingSignSoundGroup) {
		this.hangingSignSoundGroup = hangingSignSoundGroup;
		return this;
	}

	/**
	 * Sets this wood type's fence gate close sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14861}.
	 *
	 * @return this builder for chaining
	 */
	public WoodTypeBuilder fenceGateCloseSound(class_3414 fenceGateCloseSound) {
		this.fenceGateCloseSound = fenceGateCloseSound;
		return this;
	}

	/**
	 * Sets this wood type's fence gate open sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14766}.
	 *
	 * @return this builder for chaining
	 */
	public WoodTypeBuilder fenceGateOpenSound(class_3414 fenceGateOpenSound) {
		this.fenceGateOpenSound = fenceGateOpenSound;
		return this;
	}

	/**
	 * Creates a new {@link WoodTypeBuilder} that copies all of another builder's values.
	 *
	 * @param builder the {@link WoodTypeBuilder} whose values are to be copied
	 *
	 * @return the created copy
	 */
	public static WoodTypeBuilder copyOf(WoodTypeBuilder builder) {
		WoodTypeBuilder copy = new WoodTypeBuilder();
		copy.soundGroup(builder.soundGroup);
		copy.hangingSignSoundGroup(builder.hangingSignSoundGroup);
		copy.fenceGateCloseSound(builder.fenceGateCloseSound);
		copy.fenceGateOpenSound(builder.fenceGateOpenSound);
		return copy;
	}

	/**
	 * Creates a new {@link WoodTypeBuilder} that copies all of another wood type's values.
	 *
	 * @param woodType the {@link class_4719} whose values are to be copied
	 *
	 * @return the created copy
	 */
	public static WoodTypeBuilder copyOf(class_4719 woodType) {
		WoodTypeBuilder copy = new WoodTypeBuilder();
		copy.soundGroup(woodType.comp_1301());
		copy.hangingSignSoundGroup(woodType.comp_1302());
		copy.fenceGateCloseSound(woodType.comp_1303());
		copy.fenceGateOpenSound(woodType.comp_1304());
		return copy;
	}

	/**
	 * Builds and registers a {@link class_4719} from this builder's values.
	 *
	 * <p>Alternatively, you can use {@link #build(class_2960, class_8177)} to build without registering.
	 * <br>Then {@link class_4719#method_24027(class_4719)} can be used to register it later.
	 *
	 * @param id the id for the built {@link class_4719}
	 * @param setType the {@link class_8177} for the built {@link class_4719}
	 *
	 * @return the built and registered {@link class_4719}
	 */
	public class_4719 register(class_2960 id, class_8177 setType) {
		return class_4719.method_24027(this.build(id, setType));
	}

	/**
	 * Builds a {@link class_4719} from this builder's values without registering it.
	 *
	 * <p>Use {@link class_4719#method_24027(class_4719)} to register it later.
	 * <br>Alternatively, you can use {@link #register(class_2960, class_8177)} to build and register it now.
	 *
	 * @param id the id for the built {@link class_4719}
	 * @param setType the {@link class_8177} for the built {@link class_4719}
	 *
	 * @return the built {@link class_4719}
	 */
	public class_4719 build(class_2960 id, class_8177 setType) {
		return new class_4719(id.toString(), setType, soundGroup, hangingSignSoundGroup, fenceGateCloseSound, fenceGateOpenSound);
	}
}
