/*
 * Copyright (c) 2016, 2017, 2018, 2019 FabricMC
 * Copyright 2023 The Quilt Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.fabricmc.fabric.api.object.builder.v1.block.type;

import net.minecraft.class_2498;
import net.minecraft.class_2960;
import net.minecraft.class_3414;
import net.minecraft.class_3417;
import net.minecraft.class_8177;

/**
 * This class allows easy creation of {@link class_8177}s.
 *
 * <p>A {@link class_8177} is used to tell the game various properties of related blocks, such as what sounds they should use.
 *
 * @see WoodTypeBuilder
 */
public final class BlockSetTypeBuilder {
	private boolean openableByHand = true;
	private boolean openableByWindCharge = true;
	private boolean buttonActivatedByArrows = true;
	private class_8177.class_2441 pressurePlateActivationRule = class_8177.class_2441.field_11361;
	private class_2498 soundGroup = class_2498.field_11547;
	private class_3414 doorCloseSound = class_3417.field_14541;
	private class_3414 doorOpenSound = class_3417.field_14664;
	private class_3414 trapdoorCloseSound = class_3417.field_15080;
	private class_3414 trapdoorOpenSound = class_3417.field_14932;
	private class_3414 pressurePlateClickOffSound = class_3417.field_15002;
	private class_3414 pressurePlateClickOnSound = class_3417.field_14961;
	private class_3414 buttonClickOffSound = class_3417.field_15105;
	private class_3414 buttonClickOnSound = class_3417.field_14699;

	/**
	 * Sets whether this block set type's door and trapdoor can be opened by hand.
	 *
	 * <p>Defaults to {@code true}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder openableByHand(boolean openableByHand) {
		this.openableByHand = openableByHand;
		return this;
	}

	/**
	 * Sets whether this block set type's door and trapdoor can be opened by wind charge.
	 *
	 * <p>Defaults to {@code true}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder openableByWindCharge(boolean openableByWindCharge) {
		this.openableByWindCharge = openableByWindCharge;
		return this;
	}

	/**
	 * Sets whether this block set type's button can be activated by an arrow.
	 *
	 * <p>Defaults to {@code true}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder buttonActivatedByArrows(boolean buttonActivatedByArrows) {
		this.buttonActivatedByArrows = buttonActivatedByArrows;
		return this;
	}

	/**
	 * Sets this block set type's pressure plate {@link class_8177.class_2441}.
	 *
	 * <p>Defaults to {@link class_8177.class_2441#field_11361}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder pressurePlateActivationRule(class_8177.class_2441 activationRule) {
		this.pressurePlateActivationRule = activationRule;
		return this;
	}

	/**
	 * Sets this block set type's sound group.
	 *
	 * <p>Defaults to {@link class_2498#field_11547}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder soundGroup(class_2498 soundGroup) {
		this.soundGroup = soundGroup;
		return this;
	}

	/**
	 * Sets this block set type's door close sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14541}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder doorCloseSound(class_3414 doorCloseSound) {
		this.doorCloseSound = doorCloseSound;
		return this;
	}

	/**
	 * Sets this block set type's door open sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14664}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder doorOpenSound(class_3414 doorOpenSound) {
		this.doorOpenSound = doorOpenSound;
		return this;
	}

	/**
	 * Sets this block set type's trapdoor close sound.
	 *
	 * <p>Defaults to {@link class_3417#field_15080}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder trapdoorCloseSound(class_3414 trapdoorCloseSound) {
		this.trapdoorCloseSound = trapdoorCloseSound;
		return this;
	}

	/**
	 * Sets this block set type's trapdoor open sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14932}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder trapdoorOpenSound(class_3414 trapdoorOpenSound) {
		this.trapdoorOpenSound = trapdoorOpenSound;
		return this;
	}

	/**
	 * Sets this block set type's pressure plate click off sound.
	 *
	 * <p>Defaults to {@link class_3417#field_15002}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder pressurePlateClickOffSound(class_3414 pressurePlateClickOffSound) {
		this.pressurePlateClickOffSound = pressurePlateClickOffSound;
		return this;
	}

	/**
	 * Sets this block set type's pressure plate click on sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14961}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder pressurePlateClickOnSound(class_3414 pressurePlateClickOnSound) {
		this.pressurePlateClickOnSound = pressurePlateClickOnSound;
		return this;
	}

	/**
	 * Sets this block set type's button click off sound.
	 *
	 * <p>Defaults to {@link class_3417#field_15105}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder buttonClickOffSound(class_3414 buttonClickOffSound) {
		this.buttonClickOffSound = buttonClickOffSound;
		return this;
	}

	/**
	 * Sets this block set type's button click on sound.
	 *
	 * <p>Defaults to {@link class_3417#field_14699}.
	 *
	 * @return this builder for chaining
	 */
	public BlockSetTypeBuilder buttonClickOnSound(class_3414 buttonClickOnSound) {
		this.buttonClickOnSound = buttonClickOnSound;
		return this;
	}

	/**
	 * Creates a new {@link BlockSetTypeBuilder} that copies all of another builder's values.
	 *
	 * @param builder the {@link BlockSetTypeBuilder} whose values are to be copied
	 *
	 * @return the created copy
	 */
	public static BlockSetTypeBuilder copyOf(BlockSetTypeBuilder builder) {
		BlockSetTypeBuilder copy = new BlockSetTypeBuilder();
		copy.openableByHand(builder.openableByHand);
		copy.openableByWindCharge(builder.openableByWindCharge);
		copy.buttonActivatedByArrows(builder.buttonActivatedByArrows);
		copy.pressurePlateActivationRule(builder.pressurePlateActivationRule);
		copy.soundGroup(builder.soundGroup);
		copy.doorCloseSound(builder.doorCloseSound);
		copy.doorOpenSound(builder.doorOpenSound);
		copy.trapdoorCloseSound(builder.trapdoorCloseSound);
		copy.trapdoorOpenSound(builder.trapdoorOpenSound);
		copy.pressurePlateClickOffSound(builder.pressurePlateClickOffSound);
		copy.pressurePlateClickOnSound(builder.pressurePlateClickOnSound);
		copy.buttonClickOffSound(builder.buttonClickOffSound);
		copy.buttonClickOnSound(builder.buttonClickOnSound);
		return copy;
	}

	/**
	 * Creates a new {@link BlockSetTypeBuilder} that copies all of another block set type's values.
	 *
	 * @param setType the {@link class_8177} whose values are to be copied
	 *
	 * @return the created copy
	 */
	public static BlockSetTypeBuilder copyOf(class_8177 setType) {
		BlockSetTypeBuilder copy = new BlockSetTypeBuilder();
		copy.openableByHand(setType.comp_1471());
		copy.openableByWindCharge(setType.comp_2112());
		copy.buttonActivatedByArrows(setType.comp_1972());
		copy.pressurePlateActivationRule(setType.comp_1973());
		copy.soundGroup(setType.comp_1290());
		copy.doorCloseSound(setType.comp_1291());
		copy.doorOpenSound(setType.comp_1292());
		copy.trapdoorCloseSound(setType.comp_1293());
		copy.trapdoorOpenSound(setType.comp_1294());
		copy.pressurePlateClickOffSound(setType.comp_1295());
		copy.pressurePlateClickOnSound(setType.comp_1296());
		copy.buttonClickOffSound(setType.comp_1297());
		copy.buttonClickOnSound(setType.comp_1298());
		return copy;
	}

	/**
	 * Builds and registers a {@link class_8177} from this builder's values.
	 *
	 * <p>Alternatively, you can use {@link #build(class_2960)} to build without registering.
	 * <br>Then {@link class_8177#method_49233(class_8177)} can be used to register it later.
	 *
	 * @param id the id for the built {@link class_8177}
	 *
	 * @return the built and registered {@link class_8177}
	 */
	public class_8177 register(class_2960 id) {
		return class_8177.method_49233(this.build(id));
	}

	/**
	 * Builds a {@link class_8177} from this builder's values without registering it.
	 *
	 * <p>Use {@link class_8177#method_49233(class_8177)} to register it later.
	 * <br>Alternatively, you can use {@link #register(class_2960)} to build and register it now.
	 *
	 * @param id the id for the built {@link class_8177}
	 *
	 * @return the built {@link class_8177}
	 */
	public class_8177 build(class_2960 id) {
		return new class_8177(id.toString(), openableByHand, openableByWindCharge, buttonActivatedByArrows, pressurePlateActivationRule, soundGroup, doorCloseSound, doorOpenSound, trapdoorCloseSound, trapdoorOpenSound, pressurePlateClickOffSound, pressurePlateClickOnSound, buttonClickOffSound, buttonClickOnSound);
	}
}
